#!/bin/bash
###############################################################################
# NetControl.command – Netzwerkdiagnose mit Icons und Interface-Zuordnung
# Inklusive Erklärung für Thunderbolt/USB-C-Interfaces
###############################################################################

SCRIPT_DIR="$(dirname "$0")"
REPORT_DIR="$HOME/Desktop/NetControl"
REPORT_FILE="$REPORT_DIR/report.md"
VENV_DIR="$REPORT_DIR/venv"

mkdir -p "$REPORT_DIR"
line() { echo -e "\n---\n" >> "$REPORT_FILE"; }

echo "# Netzwerkdiagnose – $(date)" > "$REPORT_FILE"
line
echo "## 1️⃣ Netzwerk-Adapter (sortiert)" >> "$REPORT_FILE"

declare -a eth_if wlan_if other_if

# Hardwareports sammeln und klassifizieren
while IFS= read -r line; do
    if [[ "$line" == Hardware* ]]; then
        port=$(echo "$line" | cut -d: -f2- | xargs)
    elif [[ "$line" == Device* ]]; then
        dev=$(echo "$line" | awk '{print $2}')
        if [[ "$port" =~ Wi[-]*Fi|AirPort ]]; then
            wlan_if+=("$dev")
        elif [[ "$port" =~ Ethernet ]]; then
            eth_if+=("$dev")
        else
            other_if+=("$dev")
        fi
    fi
done < <(networksetup -listallhardwareports)

print_group() {
    local title=$1; shift
    local list=("$@")
    [[ ${#list[@]} -eq 0 ]] && return
    echo "### $title" >> "$REPORT_FILE"
    for iface in "${list[@]}"; do
        ip=$(ipconfig getifaddr "$iface" 2>/dev/null)
        if ifconfig "$iface" | grep -q "status: active"; then
            if [[ -n "$ip" ]]; then
                icon="🟢✅"   # aktiv + IP
            else
                icon="🟡⚠️"   # aktiv aber keine IP
            fi
        else
            icon="🔴❌"       # inaktiv
        fi
        speed=$(ifconfig "$iface" | grep -Eo "media:.*" | sed 's/media: //')
        echo "$icon **Interface:** $iface" >> "$REPORT_FILE"
        echo "- IP-Adresse: ${ip:-keine}" >> "$REPORT_FILE"
        echo "- Medieninfo: ${speed:-unbekannt}" >> "$REPORT_FILE"
        echo "" >> "$REPORT_FILE"
    done
}

print_group "Ethernet" "${eth_if[@]}"
print_group "WLAN"     "${wlan_if[@]}"
print_group "Andere"   "${other_if[@]}"

# --- Erklärung für spezielle Interfaces -------------------------------------
line
cat >> "$REPORT_FILE" <<'EXPLAIN'
### ℹ️ Hinweis zu bestimmten Interfaces

**en2 / en3 / en4 / en5**  
Reservierte Slots für **Thunderbolt- oder USB-C-Adapter**, z. B. wenn du ein Thunderbolt-Ethernet- oder USB-C-Dock anschließt.  
Ohne aktives Gerät bleiben sie *down* und ohne IP.  
✅ Thunderbolt 3/4 bzw. USB-C 3.x Adapter

**bridge0**  
Virtuelles Interface, das macOS z. B. für **Internetfreigabe**, **Sidecar**, **Hypervisor** oder **Content Caching** anlegt.  
_Rein virtuell, keine physische Verbindung._
EXPLAIN

# --- 2. Geschwindigkeit -----------------------------------------------------
line
echo "## 2️⃣ Geschwindigkeitstest" >> "$REPORT_FILE"
gateway=$(route -n get default 2>/dev/null | awk '/gateway/ {print $2}')
echo "- Gateway: ${gateway:-unbekannt}" >> "$REPORT_FILE"
echo "\n**Ping zu 8.8.8.8 (Google DNS)**" >> "$REPORT_FILE"
ping -c 5 8.8.8.8 | tail -n 3 >> "$REPORT_FILE"

if command -v python3 >/dev/null 2>&1; then
    echo "\n**Erweiterter Speedtest (Python)**" >> "$REPORT_FILE"
    if [ ! -d "$VENV_DIR" ]; then
        echo "(Lege virtuelle Umgebung an …)" >> "$REPORT_FILE"
        python3 -m venv "$VENV_DIR"
    fi
    source "$VENV_DIR/bin/activate"
    python3 -m pip install --quiet --upgrade pip
    python3 -m pip install --quiet speedtest-cli
    python3 - <<'PYCODE' >> "$REPORT_FILE"
import speedtest
st = speedtest.Speedtest()
st.get_best_server()
print(f"- Download: {st.download()/1_000_000:.2f} Mbit/s")
print(f"- Upload:   {st.upload()/1_000_000:.2f} Mbit/s")
print(f"- Ping:     {st.results.ping:.2f} ms")
PYCODE
    deactivate
else
    echo "_Python nicht gefunden – erweiterter Speedtest übersprungen_" >> "$REPORT_FILE"
fi

# --- 3. Geräte im lokalen Netz ----------------------------------------------
line
echo "## 3️⃣ Geräte im lokalen Netz" >> "$REPORT_FILE"
for iface in "${eth_if[@]}" "${wlan_if[@]}"; do
    if ifconfig "$iface" | grep -q "status: active"; then
        echo "### Geräte über $iface" >> "$REPORT_FILE"
        sudo ping -c3 -I "$iface" 255.255.255.255 >/dev/null 2>&1
        arp -i "$iface" -a >> "$REPORT_FILE"
        echo "" >> "$REPORT_FILE"
    fi
done

line
echo "✅ Bericht erstellt: $REPORT_FILE"
open -a "TextEdit" "$REPORT_FILE"
exit 0